const BADGE = {
  on: { text: 'ON', color: '#3b82f6' },
  off: { text: '', color: '#000000' },
  connecting: { text: '…', color: '#F59E0B' },
  error: { text: '!', color: '#ef4444' },
}

let relayWs = null
let relayConnectPromise = null
let nextSession = 1
const tabs = new Map()
const pending = new Map()

async function getGatewayConfig() {
  const stored = await chrome.storage.local.get(['gatewayUrl', 'gatewayToken'])
  return {
    url: (stored.gatewayUrl || 'https://os.myvantageai.com').replace(/\/$/, ''),
    token: stored.gatewayToken || '8d63b2600be30e5c3d81083ef13d3184e963b47db38233d1'
  }
}

function setBadge(tabId, kind) {
  const cfg = BADGE[kind]
  void chrome.action.setBadgeText({ tabId, text: cfg.text })
  void chrome.action.setBadgeBackgroundColor({ tabId, color: cfg.color })
}

async function ensureRelayConnection() {
  if (relayWs && relayWs.readyState === WebSocket.OPEN) return
  if (relayConnectPromise) return await relayConnectPromise

  relayConnectPromise = (async () => {
    const config = await getGatewayConfig()
    const wsUrl = config.url.replace(/^http/, 'ws') + '/extension'

    console.log('[Alfred] Connecting to relay:', wsUrl)
    const ws = new WebSocket(wsUrl)
    relayWs = ws

    await new Promise((resolve, reject) => {
      const t = setTimeout(() => { ws.close(); reject(new Error('Relay handshake timeout')) }, 15000)
      ws.onopen = () => {
        clearTimeout(t)
        ws.send(JSON.stringify({
            method: 'connect',
            params: {
                auth: { token: config.token },
                client: {
                    id: 'openclaw-control-ui',
                    displayName: 'Vantage OS Extension',
                    version: '2026.2.1',
                    platform: 'chrome-extension',
                    mode: 'ui'
                }
            }
        }))
        resolve()
      }
      ws.onerror = () => { clearTimeout(t); reject(new Error('Connection failed')) }
      ws.onclose = (e) => { 
          clearTimeout(t)
          reject(new Error(`Disconnected: ${e.code}`)) 
      }
    })

    ws.onmessage = (event) => {
        let msg; try { msg = JSON.parse(event.data) } catch (e) { return }
        console.log('[Alfred] Incoming message:', msg.method || msg.id)
        if (msg.method === 'ping') ws.send(JSON.stringify({ method: 'pong' }))
        if (msg.method === 'forwardCDPCommand') handleCommand(msg)
    }
  })()

  try { await relayConnectPromise } finally { relayConnectPromise = null }
}

async function handleCommand(msg) {
    console.log('[Alfred] Handling CDP command:', msg.params.method)
    try {
        const tabId = Array.from(tabs.keys())[0]
        if (!tabId) throw new Error('No attached tab found')
        const result = await chrome.debugger.sendCommand({ tabId }, msg.params.method, msg.params.params)
        console.log('[Alfred] CDP result success:', msg.params.method)
        relayWs.send(JSON.stringify({ type: 'res', id: msg.id, result }))
    } catch (err) {
        console.error('[Alfred] CDP error:', err.message)
        relayWs.send(JSON.stringify({ type: 'res', id: msg.id, error: err.message }))
    }
}

async function attachTab(tabId) {
  await chrome.debugger.attach({ tabId }, '1.3')
  const info = await chrome.debugger.sendCommand({ tabId }, 'Target.getTargetInfo')
  const sessionId = `cb-tab-${nextSession++}`
  tabs.set(tabId, { sessionId, targetId: info.targetInfo.targetId })
  
  relayWs.send(JSON.stringify({ 
    method: 'forwardCDPEvent', 
    params: { 
        method: 'Target.attachedToTarget', 
        params: { 
            sessionId, 
            targetInfo: { ...info.targetInfo, attached: true },
            waitingForDebugger: false
        } 
    } 
  }))
  setBadge(tabId, 'on')
}

chrome.action.onClicked.addListener(async () => {
  const [active] = await chrome.tabs.query({ active: true, currentWindow: true })
  if (!active?.id) return
  if (tabs.has(active.id)) {
      await chrome.debugger.detach({ tabId: active.id }).catch(() => {})
      tabs.delete(active.id)
      setBadge(active.id, 'off')
      return
  }
  setBadge(active.id, 'connecting')
  try { 
      await ensureRelayConnection()
      await attachTab(active.id) 
  }
  catch (err) { 
      console.error('[Alfred] Interaction error:', err.message)
      setBadge(active.id, 'error')
      alert(`Alfred Connection Logic Fault: ${err.message}`) 
  }
})
